/*:
 * @target MZ
 * @plugindesc アクターの装備スロットを一時ロック/解除（セーブ後も保持、最強ボタンも安全）
 * @author ChatGPT
 *
 * @help
 * 使い方（イベントのスクリプト欄）:
 *   $gameActors.actor(1).lockEquipSlot(1);   // ロック（例: 盾=1）
 *   $gameActors.actor(1).unlockEquipSlot(1); // 解除
 *   $gameActors.actor(1).clearEquipLocks();  // 全解除
 *
 * スロット番号（デフォルト）: 0=武器,1=盾,2=頭,3=体,4=装飾
 */

(() => {
  // 装備かどうかの判定（MZには isEquip がない）
  const isEquip = it => DataManager.isWeapon(it) || DataManager.isArmor(it);

  // ---- 永続化安全な配列ベース ----
  Game_Actor.prototype._ensureLockArray = function() {
    if (!Array.isArray(this._lockedEquipSlotsArr)) this._lockedEquipSlotsArr = [];
    return this._lockedEquipSlotsArr;
  };
  Game_Actor.prototype.isEquipSlotLocked = function(slotId) {
    const arr = this._ensureLockArray();
    return arr.includes(Number(slotId));
  };
  Game_Actor.prototype.lockEquipSlot = function(slotId) {
    const arr = this._ensureLockArray();
    slotId = Number(slotId);
    if (!arr.includes(slotId)) arr.push(slotId);
  };
  Game_Actor.prototype.unlockEquipSlot = function(slotId) {
    const arr = this._ensureLockArray();
    const i = arr.indexOf(Number(slotId));
    if (i >= 0) arr.splice(i, 1);
  };
  Game_Actor.prototype.clearEquipLocks = function() {
    this._lockedEquipSlotsArr = [];
  };

  // ---- 変更系APIのガード ----
  const _changeEquip = Game_Actor.prototype.changeEquip;
  Game_Actor.prototype.changeEquip = function(slotId, item) {
    if (this.isEquipSlotLocked(slotId)) {
      SoundManager.playBuzzer();
      return;
    }
    _changeEquip.call(this, slotId, item);
  };

  // 「強制変更」もブロック（他プラグイン対策）
  const _forceChangeEquip = Game_Actor.prototype.forceChangeEquip;
  Game_Actor.prototype.forceChangeEquip = function(slotId, item) {
    if (this.isEquipSlotLocked(slotId)) {
      SoundManager.playBuzzer();
      return;
    }
    _forceChangeEquip.call(this, slotId, item);
  };

  // 全装備外し：ロック枠は残す
  const _clearEquipments = Game_Actor.prototype.clearEquipments;
  Game_Actor.prototype.clearEquipments = function() {
    for (let i = 0; i < this.equipSlots().length; i++) {
      if (!this.isEquipSlotLocked(i)) this.changeEquip(i, null);
    }
  };

  // --- 最強（最適化）：ロック枠は触らず、良い候補がある時だけ更新 ---
  const _optimizeEquipments = Game_Actor.prototype.optimizeEquipments;
  Game_Actor.prototype.optimizeEquipments = function() {
    const slots = this.equipSlots();

    for (let i = 0; i < slots.length; i++) {
      if (this.isEquipSlotLocked(i)) continue; // ロックは完全スキップ

      const etypeId = slots[i];
      const current = this.equips()[i];

      // 現在装備がそのまま有効ならそれを基準にする（＝空にしない）
      let best =
        current && this.canEquip(current) && current.etypeId === etypeId
          ? current
          : null;

      // 所持品から同etypeの候補を探索
      for (const item of $gameParty.items()) {
        if (!isEquip(item)) continue;
        if (item.etypeId !== etypeId) continue;
        if (!this.canEquip(item)) continue;

        if (!best || this.calcEquipItemPerformance(item) > this.calcEquipItemPerformance(best)) {
          best = item;
        }
      }

      // ベストが現在と違う時だけ更新（外さない方針）
      if (best !== current) {
        this.changeEquip(i, best); // bestがnullでも、ここまでで基本的にcurrentが優先されている
      }
    }
  };

  // 装備画面でロック枠を選べないように（グレーアウト）
  const _isEnabled = Window_EquipSlot.prototype.isEnabled;
  Window_EquipSlot.prototype.isEnabled = function(index) {
    const base = _isEnabled.call(this, index);
    if (!this._actor) return base;
    return this._actor.isEquipSlotLocked(index) ? false : base;
  };

  // 職業変更等の整理処理の保険：ロック枠は外さない
  const _releaseUnequippableItems = Game_Actor.prototype.releaseUnequippableItems;
  Game_Actor.prototype.releaseUnequippableItems = function(forcing) {
    for (let i = 0; i < this.equipSlots().length; i++) {
      if (this.isEquipSlotLocked(i)) continue; // ロックは保持
      const item = this.equips()[i];
      if (!item) continue;
      const etypeId = this.equipSlots()[i];
      if (!this.canEquip(item) || item.etypeId !== etypeId) {
        this.changeEquip(i, null);
      }
    }
  };
})();
